# 標準モジュール
import os
import datetime
import base64

# 外部モジュール
from robot.libraries.BuiltIn import BuiltIn

# 独自モジュール
import model.constant as const
from test_state_request import check_license_setting_status, get_msg_by_view_lang  # noqa
import model.common as common
import model.pkg_utils.img_utils as img_utils
import model.pkg_img_process.img_process_controller as img_process_controller
import model.pkg_action_chanins.common as action_chanins_common


screen_name = "Empty"
# 予約語
reserved_word_list_ja = [
    '[現在の日付時刻]',
    '[現在の年]',
    '[現在の月]',
    '[現在の日]',
    '[現在の時刻]',
    '[現在の秒数]'
]
reserved_word_list_en = [
    '[Current_date_time]',
    '[Current_year]',
    '[Current_month]',
    '[Current_day]',
    '[Current_time]',
    '[Current_seconds]'
]
# 予約語フォーマット
reserved_word_format_list = [
    '%Y/%m/%d %H:%M:%S',
    '%Y',
    '%m',
    '%d',
    '%H:%M',
    '%S'
]

# デバッグ使用変数
debug_variable = ""

# 一時的保存用のグローバル変数
TDASH_GET_VALUE = r"${TDASH_GET_VALUE}"


def set_debug_variable(sckey, value):
    """
    デバッグ用変数設定
    """
    global debug_variable
    debug_variable = value
    # NOTE set_local_variableだと書き変わらない
    BuiltIn().set_global_variable('${' + sckey + '}', {"xpath": value})


def common_valtes_check_variable_exists(var_str):
    """
    デバッグ用変数設定
    """
    var_dict = BuiltIn().get_variables()
    res = False
    if var_str in var_dict:
        res = True
    return res


###################
# ロケーション関連 #
###################
def valtes_get_resource_value(screen_name, element_name):
    """
    リソースファイルから値を取得する。

    Parameters
    ------------------
    screen_name : string
        画面名。
    element_name : string
        要素名。

    Returns
    ------------------
    results_value : string
        リソースファイルに定義された値。
    """
    results_dict = get_resouce_dict(screen_name, element_name)
    results_value = results_dict.get('xpath', "")

    BuiltIn().log(f"${screen_name}.{element_name}={results_value}")

    return results_value


def valtes_get_img_location(screen_name, element_name):
    """
    リソースファイルから値を取得する。

    Parameters
    ------------------
    screen : string
        画面名。
    item : string
        項目名。

    Returns
    ------------------
    results_value : string
        リソースファイルに定義された値。
    """
    # choose_location参照
    results_dict = get_resouce_dict(screen_name, element_name)

    # 無料プランで紛れ込んだ場合、xpathで実行
    if check_license_setting_status().get('is_free_plan', True):
        xpath = results_dict.get('xpath', "")
        log_msg = f"${screen_name}.{element_name}={xpath}"
        BuiltIn().log(log_msg)
        return None, xpath

    # 画像からロケーションを取得する
    element_img_path = results_dict.get('element_img_path', "")
    candidate_position_list, candidate_xpath_list = get_candidates_by_img(
        element_img_path)

    # 座標XPathの決定
    element_position = candidate_position_list[0]
    element_xpath = candidate_xpath_list[0]

    log_msg = f"${screen_name}.{element_name}={element_img_path}\n{element_position}\n{element_xpath}"
    BuiltIn().log(log_msg)

    return element_position, element_xpath


def get_resouce_dict(arg_screen_name, arg_element_name):
    """
    _page.robotからデータを取り出す

    Parameters
    ------------------
    arg_screen_name : string
        画面名
    arg_element_name : string
        要素名

    Returns
    ------------------
    results_dict : string
        リソースファイルに定義された値。
    """
    global debug_variable
    global screen_name
    results_dict = {}
    try:
        BuiltIn().log(f"{screen_name}->{arg_screen_name}")
        if screen_name != arg_screen_name:
            screen_name = arg_screen_name

        if not debug_variable:
            BuiltIn().import_resource(f"{arg_screen_name}_page.robot")

        # キーの決定
        escaped_screen_name = common.escape_rf_word(screen_name)
        escaped_arg_element_name = common.escape_rf_word(arg_element_name)
        key = "${" + f"{escaped_screen_name}.{escaped_arg_element_name}" + "}"

        # データ取得
        results_dict = BuiltIn().get_variable_value(key)

    except Exception:
        error_msg = f"Undefined:{arg_screen_name}_page={arg_screen_name}.{arg_element_name}"
        raise AssertionError(error_msg)

    return results_dict


def get_candidates_by_img(element_img_path):
    """
    要素画像からロケーション(=XPath)を取得する

    Parameters
    ------------------
    element_img_path : string
        要素画像パス

    Returns
    ------------------
    candidate_xpath_list[0] : string
        ロケーション(=XPath)の候補第一位
    """
    selelib = BuiltIn().get_library_instance("SeleniumLibrary")
    wbd = selelib.driver

    #######################################
    # 現在の画面をSSし、そのパスを取得する #
    #######################################
    execute_capture_fullsize_screenshot("current_page_img")
    page_img_full_path = img_utils.get_latest_capture()

    ####################
    # 候補座標を取得 #
    #   x: +右方向     #
    #   y: +下方向     #
    ####################
    candidate_position_list, analized_img_a_path = img_process_controller.analize_element_img(
        page_img_full_path,
        element_img_path,
    )

    if len(candidate_position_list) == 0:
        return [], []

    ###########################
    # 候補XPathを座標から取得 #
    ###########################
    candidate_xpath_list = []
    for candidate_position in candidate_position_list:
        # 座標からロケーションを取得
        x = candidate_position[0]
        y = candidate_position[1]
        element = action_chanins_common.get_element_by_position(wbd, x, y)
        temp_location = ""
        try:
            temp_location = common.get_xpath_by_element(element)
        except Exception:
            msg = get_msg_by_view_lang(
                "test_run_process.not_exist_xpath")
            print(msg)

        candidate_xpath_list += [temp_location]

    return candidate_position_list, candidate_xpath_list


#########################
# スクリーンショット関係 #
#########################
def execute_capture_fullsize_screenshot(value):
    """
    画面のフルスクリーンショットを撮る。
    ※レポートに画像を表示する場合: SS_ROBOT_KEYWORD_LIST に BeasicOperation を追加

    Parameters
    ------------------
    value : string
        ファイル名、保存先はlogファイルと同じディレクトリ。
    """
    # フォルダ作成は絶対パスで
    report_folder = BuiltIn().get_variable_value('${OUTPUT DIR}')
    ss_folder_a_path = os.path.normpath(
        os.path.join(
            report_folder,
            const.SS_FOLDER_NAME
        )
    )
    os.makedirs(ss_folder_a_path, exist_ok=True)

    # フルスクショ作成
    image = get_fullsize_screenshot()

    # スクショ名
    ss_file_name = "./{0}-{1}.png".format(
        value,
        datetime.datetime.now().strftime('%Y%m%d-%H%M%S%f'),
    )

    # 画像を出力 ※絶対パス( or T-DASHルートからの相対パス )
    ss_a_path = os.path.normpath(
        os.path.join(
            ss_folder_a_path,
            ss_file_name,
        )
    )
    with open(ss_a_path, 'bw') as f:
        f.write(image)

    # レポートには相対パスで出力
    ss_r_path = os.path.normpath(
        os.path.join(
            const.SS_FOLDER_NAME,
            ss_file_name,
        )
    )
    img_utils.set_img_in_xml(ss_r_path)

    return True


def get_fullsize_screenshot():
    """
    フルサイズのスクリーンショットを取得する
    """
    driver = BuiltIn().get_library_instance('SeleniumLibrary').driver
    cdp_browser = {'chrome', 'msedge'}
    browser_name = driver.capabilities['browserName']

    image = None
    if browser_name in cdp_browser:
        width = driver.execute_script("return document.body.scrollWidth;")
        height = driver.execute_script("return document.body.scrollHeight;")
        viewport = {
            "x": 0,
            "y": 0,
            "width": width,
            "height": height,
            "scale": 1
        }
        # Chrome Devtools Protocolコマンドを実行し、取得できるBase64形式の画像データをデコードしてファイルに保存
        image_base64 = driver.execute_cdp_cmd(
            "Page.captureScreenshot", {"clip": viewport, "captureBeyondViewport": True})
        image = base64.b64decode(image_base64["data"])
    else:
        image_base64 = driver.get_full_page_screenshot_as_base64()
        image = base64.b64decode(image_base64)

    return image


##########
# その他 #
##########
def valtes_reserved_word(value):
    """
    テストパターン自動生成(フェーズ１)

    Parameters
    ------------------
    value : string
        ファイル名、保存先はlogファイルと同じディレクトリ。
        入力値。
    Returns
    ------------------
    value : string
        予約語生成文字列
    """
    value = str(value)
    i = 0
    while len(reserved_word_list_ja) > i:
        if reserved_word_list_ja[i] in value:
            value = replace_reserved_word(
                value,
                reserved_word_list_ja[i],
                reserved_word_format_list[i],
            )
        if reserved_word_list_en[i] in value:
            value = replace_reserved_word(
                value,
                reserved_word_list_en[i],
                reserved_word_format_list[i],
            )
        i += 1

    return value


def replace_reserved_word(setting_value, reserved_word, reserved_word_format):
    """
    予約語の置き換えを行う

    Parameters
    ------------------
    value : string
        ファイル名、保存先はlogファイルと同じディレクトリ。
        入力値。
    Returns
    ------------------
    value : string
        予約語生成文字列
    """
    # 置き換え
    cur_time = datetime.datetime.now().strftime(
        reserved_word_format)
    setting_value = setting_value.replace(
        reserved_word, cur_time)

    # レポートに出力
    msg = f"<current_data_time reserved_word='{reserved_word}' cur_time='{cur_time}'>run time = {cur_time}</current_data_time>"
    BuiltIn().log(msg, html=True)

    return setting_value


def valtes_not_support(libname):
    liblist = {"ValtesAppiumLibrary": "Mobile Browser",
               "ValtesSeleniumLibrary": "PC Browser"}
    errormsg = "This action is not supported by {}".format(liblist[libname])
    raise Exception(errormsg)


def common_valtes_save_value(value):
    """
    一時的変数に保存された値を指定の変数に保存する

    Parameters
    ------------------
    value : string
        変数名。
    """
    BuiltIn().set_global_variable('${' + value + '}', BuiltIn().get_variable_value(TDASH_GET_VALUE))
